﻿using System;
using System.Collections;
using System.Collections.Generic;
using TMPro;
using UnityEngine;
using UnityEngine.EventSystems;
using UnityEngine.SceneManagement;

public class ResultMenu : MonoBehaviour
{
    private GameObject _BackgroundPanel;

    private void Awake()
    {
        //Subscribe to events
        InputManager.Instance.InputStateChanged += OnInputStateChanged;
        _BackgroundPanel = gameObject.FindChild("BackgroundPanel");
        _BackgroundPanel.GetComponent<PhaseAnimation>().BounceInCompleted += ResultMenu_BounceInCompleted;
        _BackgroundPanel.GetComponent<PhaseAnimation>().BounceOutCompleted += ResultMenu_BounceOutCompleted;
    }

    private void ResultMenu_BounceInCompleted(object sender, System.EventArgs e)
    {
        if (InputManager.Instance.InputState == Constants.InputState.Controller)
        {
            //We've just changed to a controller so focus on the play again button
            EventSystem.current.SetSelectedGameObject(null);
            EventSystem.current.SetSelectedGameObject(_BackgroundPanel.FindChild("PlayAgainButton"));
        }

        //Bouncing is done so enable the UI
        gameObject.Enable();
    }

    private void ResultMenu_BounceOutCompleted(object sender, System.EventArgs e)
    {
        //Bouncing is done so hide the result UI
        //If bounce out was called, we must have hit replay so call play again
        GameController.Instance.ResultScreen.SetActive(false);
        GameController.Instance.PlayAgain();
    }

    private void OnInputStateChanged(object sender, InputManager.InputStateChangedEventArgs e)
    {
        if (e.NewState == Constants.InputState.Controller && gameObject.activeSelf)
        {
            //We've just changed to a controller so focus on the play again button
            EventSystem.current.SetSelectedGameObject(null);
            EventSystem.current.SetSelectedGameObject(_BackgroundPanel.FindChild("PlayAgainButton"));
            e.Handled = true;
        }
    }

    private void OnDestroy()
    {
        //Unsubscribe from events
        InputManager.Instance.InputStateChanged -= OnInputStateChanged;
        _BackgroundPanel.GetComponent<PhaseAnimation>().BounceInCompleted -= ResultMenu_BounceInCompleted;
        _BackgroundPanel.GetComponent<PhaseAnimation>().BounceOutCompleted -= ResultMenu_BounceOutCompleted;
    }

    public void MoveToThis(int winner)
    {
        //We're moving to the result menu, so disable the UI
        gameObject.Disable();

        TextMeshProUGUI resultText = _BackgroundPanel.FindChild("ResultText").GetComponent<TextMeshProUGUI>();

        if (winner == 1)
        {
            //Player one won, set the text
            resultText.SetText("PLAYER 1 WINS THE GAME!");
        }

        else
        {
            //Player two won, set the text
            resultText.SetText("PLAYER 2 WINS THE GAME!");
        }

        //Begin animating the panel in and start the confetti
        _BackgroundPanel.gameObject.transform.localScale = new Vector3(0.0f, 0.0f, 1.0f);
        _BackgroundPanel.GetComponent<Animator>().Play("BounceIn", -1, 0.0f);
        _BackgroundPanel.FindChild("ConfettiLeft").GetComponent<Animator>().Play("Left", -1, 0.0f);
        _BackgroundPanel.FindChild("ConfettiRight").GetComponent<Animator>().Play("Right", -1, 0.0f);
        _BackgroundPanel.FindChild("ConfettiBottomLeft").GetComponent<Animator>().Play("Left", -1, 0.0f);
        _BackgroundPanel.FindChild("ConfettiBottomRight").GetComponent<Animator>().Play("Right", -1, 0.0f);
    }

    public void MoveToNext()
    {
        //We're moving away so disable the UI and begin animating the panel
        gameObject.Disable();
        _BackgroundPanel.GetComponent<Animator>().Play("BounceOut", -1, 0.0f);
    }

    public void QuitToMainMenu()
    {
        //We're quitting, so stop the confetti audio, reset the music volume and time scale then load the menu
        ((GameOverPhase)GameController.Instance.CurrentGamePhase).ConfettiSource.Stop();
        AudioManager.Instance.SetMusicVolume(Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.MusicVolume]));  //We need to do this so if we're paused, the audio is reset back to its proper value
        Time.timeScale = 1.0f;
        SceneManager.LoadScene("Menu");
    }
}
